<?php

  // is_blank('abcd')
  // * validate data presence
  // * uses trim() so empty spaces don't count
  // * uses === to avoid false positives
  // * better than empty() which considers "0" to be empty
  function is_blank($value) {
    return !isset($value) || trim($value) === '';
  }

  // has_presence('abcd')
  // * validate data presence
  // * reverse of is_blank()
  // * I prefer validation names with "has_"
  function has_presence($value) {
    return !is_blank($value);
  }

  // has_length_greater_than('abcd', 3)
  // * validate string length
  // * spaces count towards length
  // * use trim() if spaces should not count
  function has_length_greater_than($value, $min) {
    $length = strlen($value);
    return $length > $min;
  }

  // has_length_less_than('abcd', 5)
  // * validate string length
  // * spaces count towards length
  // * use trim() if spaces should not count
  function has_length_less_than($value, $max) {
    $length = strlen($value);
    return $length < $max;
  }

  // has_length_exactly('abcd', 4)
  // * validate string length
  // * spaces count towards length
  // * use trim() if spaces should not count
  function has_length_exactly($value, $exact) {
    $length = strlen($value);
    return $length == $exact;
  }

  // has_length('abcd', ['min' => 3, 'max' => 5])
  // * validate string length
  // * combines functions_greater_than, _less_than, _exactly
  // * spaces count towards length
  // * use trim() if spaces should not count
  function has_length($value, $options) {
    if(isset($options['min']) && !has_length_greater_than($value, $options['min'] - 1)) {
      return false;
    } elseif(isset($options['max']) && !has_length_less_than($value, $options['max'] + 1)) {
      return false;
    } elseif(isset($options['exact']) && !has_length_exactly($value, $options['exact'])) {
      return false;
    } else {
      return true;
    }
  }

  // has_inclusion_of( 5, [1,3,5,7,9] )
  // * validate inclusion in a set
  function has_inclusion_of($value, $set) {
  	return in_array($value, $set);
  }

  // has_exclusion_of( 5, [1,3,5,7,9] )
  // * validate exclusion from a set
  function has_exclusion_of($value, $set) {
    return !in_array($value, $set);
  }

  // has_string('nobody@nowhere.com', '.com')
  // * validate inclusion of character(s)
  // * strpos returns string start position or false
  // * uses !== to prevent position 0 from being considered false
  // * strpos is faster than preg_match()
  function has_string($value, $required_string) {
    return strpos($value, $required_string) !== false;
  }

  // has_valid_email_format('nobody@nowhere.com')
  // * validate correct format for email addresses
  // * format: [chars]@[chars].[2+ letters]
  // * preg_match is helpful, uses a regular expression
  //    returns 1 for a match, 0 for no match
  //    http://php.net/manual/en/function.preg-match.php
  function has_valid_email_format($value) {
    $email_regex = '/\A[A-Z0-9._%+-]+@[A-Z0-9.-]+\.[A-Z]{2,}\Z/i';
    return preg_match($email_regex, $value) === 1;
  }

// * validate correct format for password
// * has at least 1 lowercase letter
// * has at least 1 uppercase letter
// * has at least 1 number
// * has at least 1 special character consisting of (!, @, #, $, %, ^, &, *)
// * preg_match is helpful, uses a regular expression
//    returns 1 for a match, 0 for no match
//    http://php.net/manual/en/function.preg-match.php
function has_valid_password_format($value) {
    $password_regex = '/^(?=.*[a-z])(?=.*[A-Z])(?=.*[0-9])(?=.*[!@#\$%\^&\*])/';
    return preg_match($password_regex, $value) === 1;
}

function has_password_match($password, $confirm_password) {
    return $password === $confirm_password;
}

function has_unique_page_menu_name($menu_name, $current_id) {
    
    // Get database connection
    global $db;
    
    // Get page set
    $page_set = find_all_pages();
    
    while($page = mysqli_fetch_assoc($page_set)) {
        
        // Check submitted menu name against database entries
        if($page['menu_name'] === $menu_name && !($page['id'] === $current_id) ) {
            return false;
        }
    }
    
    return true;
}

function has_existing_page_record ($id) {
    
    // Get database connection
    global $db;
    
    // Get page set
    $page_set = find_all_pages();
    
    while($page = mysqli_fetch_assoc($page_set)) {
        
        // Check for existing record
        if($page['id'] === $id) {
            return true;
        } 
    }
    
    return false;
}

function has_unique_admin_username($username, $current_id) {
    
    // Get database connection
    global $db;
    
    // Get admin set
    $admin_set = find_all_admins();
    
    while($admin = mysqli_fetch_assoc($admin_set)) {
        
        // Check submitted username against database entries
        if($admin['username'] === $username && !($admin['id'] === $current_id) ) {
            return false;
        }
    }
    
    return true;
}

function has_existing_admin_record ($id) {
    
    // Get database connection
    global $db;
    
    // Get page set
    $admin_set = find_all_admins();
    
    while($admin = mysqli_fetch_assoc($admin_set)) {
        
        // Check for existing record
        if($admin['id'] === $id) {
            return true;
        } 
    }
    
    return false;
}


?>
